<?php

namespace app\models;

use Yii;
use yii\db\ActiveRecord;
use yii\web\IdentityInterface;

/**
 * @property int $id
 * @property string $full_name
 * @property string $phone
 * @property string $email
 * @property string $password_hash
 * @property string $auth_key
 * @property string $access_token
 * @property int $status
 * @property string $created_at
 * @property string $updated_at
 */
class User extends ActiveRecord implements IdentityInterface
{
    const STATUS_DELETED = 0;
    const STATUS_ACTIVE = 10;

    public $password;
    public $password_repeat;
    public $agree_terms;
    public $rememberMe = true;

    public static function tableName()
    {
        return '{{%users}}';
    }

    public function rules()
    {
        return [
            // Правила для регистрации
            [['full_name', 'phone', 'email', 'password', 'password_repeat', 'agree_terms'], 'required', 'on' => 'register'],
            ['full_name', 'match', 'pattern' => '/^[а-яА-ЯёЁ\s]+$/u', 'message' => 'ФИО должно содержать только кириллические символы и пробелы', 'on' => 'register'],
            ['phone', 'match', 'pattern' => '/^8\(\d{3}\)\d{3}-\d{2}-\d{2}$/', 'message' => 'Телефон должен быть в формате 8(XXX)XXX-XX-XX', 'on' => 'register'],
            ['email', 'email', 'on' => 'register'],
            ['email', 'unique', 'on' => 'register'],
            ['phone', 'unique', 'on' => 'register'],
            ['password', 'string', 'min' => 8, 'on' => 'register'],
            ['password_repeat', 'compare', 'compareAttribute' => 'password', 'on' => 'register'],
            ['agree_terms', 'compare', 'compareValue' => true, 'message' => 'Вы должны согласиться с правилами регистрации', 'on' => 'register'],

            // Правила для авторизации
            [['phone', 'password'], 'required', 'on' => 'login'],
            ['rememberMe', 'boolean', 'on' => 'login'],

            ['status', 'default', 'value' => self::STATUS_ACTIVE],
            ['status', 'in', 'range' => [self::STATUS_ACTIVE, self::STATUS_DELETED]],
        ];
    }

    public function scenarios()
    {
        return [
            'register' => ['full_name', 'phone', 'email', 'password', 'password_repeat', 'agree_terms'],
            'login' => ['phone', 'password', 'rememberMe'],
        ];
    }

    public function attributeLabels()
    {
        return [
            'full_name' => 'ФИО',
            'phone' => 'Телефон',
            'email' => 'E-mail',
            'password' => 'Пароль',
            'password_repeat' => 'Повторите пароль',
            'agree_terms' => 'Согласен с правилами регистрации',
            'rememberMe' => 'Запомнить меня',
        ];
    }

    public function beforeSave($insert)
    {
        if (parent::beforeSave($insert)) {
            if ($insert) {
                $this->auth_key = Yii::$app->security->generateRandomString();
            }
            if ($this->password && !empty($this->password)) {
                $this->password_hash = Yii::$app->security->generatePasswordHash($this->password);
            }
            return true;
        }
        return false;
    }

    // IdentityInterface methods
    public static function findIdentity($id)
    {
        return static::findOne(['id' => $id, 'status' => self::STATUS_ACTIVE]);
    }

    public static function findIdentityByAccessToken($token, $type = null)
    {
        return static::findOne(['access_token' => $token, 'status' => self::STATUS_ACTIVE]);
    }

    public static function findByPhone($phone)
    {
        // Очищаем телефон от всех символов, кроме цифр
        $cleanPhone = preg_replace('/[^0-9]/', '', $phone);

        // Если телефон начинается с 8 и имеет 11 цифр
        if (strlen($cleanPhone) === 11 && $cleanPhone[0] === '8') {
            // Форматируем телефон в нужный формат
            $formattedPhone = '8(' . substr($cleanPhone, 1, 3) . ')' . substr($cleanPhone, 4, 3) . '-' . substr($cleanPhone, 7, 2) . '-' . substr($cleanPhone, 9, 2);

            // Пробуем найти в отформатированном виде
            $user = static::findOne(['phone' => $formattedPhone, 'status' => self::STATUS_ACTIVE]);
            if ($user) {
                return $user;
            }

            // Пробуем найти в виде цифр
            $user = static::findOne(['phone' => $cleanPhone, 'status' => self::STATUS_ACTIVE]);
            if ($user) {
                return $user;
            }
        }

        // Пробуем найти как есть
        return static::findOne(['phone' => $phone, 'status' => self::STATUS_ACTIVE]);
    }

    public function getId()
    {
        return $this->id;
    }

    public function getAuthKey()
    {
        return $this->auth_key;
    }

    public function validateAuthKey($authKey)
    {
        return $this->getAuthKey() === $authKey;
    }

    // Метод для проверки пароля
    public function validatePassword($password)
    {
        return Yii::$app->security->validatePassword($password, $this->password_hash);
    }

    // Валидация для формы авторизации
    public function login()
    {
        if ($this->validate()) {
            return Yii::$app->user->login($this->getUser(), $this->rememberMe ? 3600*24*30 : 0);
        }
        return false;
    }

    // Получение пользователя по телефону
    public function getUser()
    {
        if ($this->_user === false) {
            $this->_user = static::findByPhone($this->phone);
        }
        return $this->_user;
    }

    private $_user = false;

    public function getApplications()
    {
        return $this->hasMany(Application::class, ['user_id' => 'id']);
    }
}